<?php
namespace YeEasyAdminNotices\V1;

class AdminNotice {
	const TYPE_SUCCESS = 'success';
	const TYPE_INFO = 'info';
	const TYPE_WARNING = 'warning';
	const TYPE_ERROR = 'error';

	const DISMISSED_OPTION_PREFIX = 'ye_is_dismissed-';
	const DELAYED_NOTICE_OPTION = 'ye_delayed_notices';

	protected $id;
	protected $content = '';
	protected $noticeType = 'success';
	protected $customCssClasses = array();

	protected $isDismissible = false;
	protected $isPersistentlyDismissible = false;

	public function __construct($id = null) {
		$this->id = $id;
	}

	/**
	 * Create a new notice.
	 *
	 * @param string $id
	 * @return AdminNotice
	 */
	public static function create($id = null) {
		return new static($id);
	}

	public function text($message) {
		$this->content = '<p>' . esc_html($message) . '<p>';
		return $this;
	}

	public function html($arbitraryHtml) {
		$this->content = '<p>' . $arbitraryHtml . '</p>';
		return $this;
	}

	public function rawHtml($arbitraryHtml) {
		$this->content = $arbitraryHtml;
		return $this;
	}

	public function type($noticeType) {
		$this->noticeType = $noticeType;
		return $this;
	}

	public function success($messageHtml = null) {
		return $this->setTypeAndMessage(self::TYPE_SUCCESS, $messageHtml);
	}

	public function info($messageHtml = null) {
		return $this->setTypeAndMessage(self::TYPE_INFO, $messageHtml);
	}

	public function warning($messageHtml = null) {
		return $this->setTypeAndMessage(self::TYPE_WARNING, $messageHtml);
	}

	public function error($messageHtml = null) {
		return $this->setTypeAndMessage(self::TYPE_ERROR, $messageHtml);
	}

	protected function setTypeAndMessage($noticeType, $messageHtml = null) {
		$this->noticeType = $noticeType;
		if (isset($messageHtml)) {
			$this->html($messageHtml);
		}
		return $this;
	}

	public function addClass($className) {
		if (is_array($className)) {
			$className = implode(' ', $className);
		}
		$this->customCssClasses[] = $className;
		return $this;
	}

	/**
	 * Make the notice dismissible.
	 *
	 * @return self
	 */
	public function dismissible() {
		$this->isDismissible = true;
		return $this;
	}

	/**
	 * When the user dismisses the notice, remember that and don't show it again.
	 * @todo: Scope: user or global
	 *
	 * @return self
	 */
	public function persistentlyDismissible() {
		if (empty($this->id)) {
			throw new \LogicException('Persistently dismissible notices must have a unique ID.');
		}

		$this->isDismissible = true;
		$this->isPersistentlyDismissible = true;

		$ajaxCallback = array($this, 'ajaxDismiss');
		if (has_action($this->getDismissActionName(), $ajaxCallback) === false) {
			add_action('wp_ajax_' . $this->getDismissActionName(), $ajaxCallback);
		}

		return $this;
	}

	/**
	 * Show the notice on the current page when appropriate.
	 */
	public function show() {
		if (did_action('admin_notices')) {
			$this->maybeOutputNotice();
		} else {
			add_action('admin_notices', array($this, 'maybeOutputNotice'));
		}
		return $this;
	}

	/**
	 * Immediately output the notice unless it has been dismissed.
	 *
	 * @internal
	 */
	public function maybeOutputNotice() {
		if ($this->isPersistentlyDismissible && $this->isDismissed()) {
			return;
		}
		$this->outputNotice();
	}

	/**
	 * Output the notice.
	 */
	public function outputNotice() {
		$attributes = array();

		if (isset($this->id)) {
			$attributes['id'] = $this->id;
		}

		$attributes['class'] = 'notice notice-' . $this->noticeType;
		if (!empty($this->customCssClasses)) {
			$attributes['class'] .= ' ' . implode(' ', $this->customCssClasses);
		}

		if ($this->isDismissible) {
			$attributes['class'] .= ' is-dismissible';
		}
		if ($this->isPersistentlyDismissible) {
			$attributes['data-ye-dismiss-nonce'] = wp_create_nonce($this->getDismissActionName());

			if (!wp_script_is('ye-dismiss-notice', 'registered')) {
				wp_register_script(
					'ye-dismiss-notice',
					plugins_url('dismiss-notice.js', __FILE__),
					array('jquery'),
					'20161126',
					true
				);
			}
			if (!wp_script_is('ye-dismiss-notice', 'enqueued') && !wp_script_is('ye-dismiss-notice', 'done')) {
				wp_enqueue_script('ye-dismiss-notice');
			}
		}

		$attributePairs = array();
		foreach($attributes as $name => $value) {
			$attributePairs[] = $name . '="' . esc_attr($value) . '"';
		}

		/** @noinspection HtmlUnknownAttribute */
		printf(
			'<div %1$s>%2$s</div>',
			implode(' ', $attributePairs),
			$this->content
		);
	}

	/**
	 * Show the notice on the next admin page that's visited by the current user.
	 * The notice will be shown only once.
	 *
	 * @return AdminNotice
	 */
	public function showOnNextPage() {
		if (!is_user_logged_in()) {
			return $this;
		}

		//Schedule the notice to appear on the next page.
		add_user_meta(
			get_current_user_id(),
			static::DELAYED_NOTICE_OPTION,
			wp_slash($this->toJson()),
			false
		);

		return $this;
	}

	/**
	 * Display delayed notices stored by showOnNextPage.
	 * @internal
	 */
	public static function _showDelayedNotices() {
		$userId = get_current_user_id();
		$notices = get_user_meta($userId, static::DELAYED_NOTICE_OPTION, false);
		if (empty($notices)) {
			return;
		}

		$myClass = get_called_class();

		foreach ($notices as $json) {
			$properties = json_decode($json, true);

			//Ignore notices created by other versions of this class.
			if (empty($properties) || empty($properties['_className']) || ($properties['_className'] !== $myClass)) {
				continue;
			}

			$notice = static::fromJson($json);
			$notice->show();

			//Only show the notice once.
			delete_user_meta($userId, static::DELAYED_NOTICE_OPTION, wp_slash($json));
		}
	}

	/**
	 * Serialize the notice as JSON.
	 *
	 * @return string
	 */
	public function toJson() {
		$data = array(
			'id' => $this->id,
			'content' => $this->content,
			'noticeType' => $this->noticeType,
			'isDismissible' => $this->isDismissible,
			'isPersistentlyDismissible' => $this->isPersistentlyDismissible,
			'customCssClasses' => $this->customCssClasses,
			'_className' => get_class($this),
		);

		return json_encode($data);
	}

	/**
	 * Load a notice from JSON.
	 *
	 * @param string $json
	 * @return AdminNotice
	 */
	public static function fromJson($json) {
		$properties = json_decode($json, true);

		$notice = new static($properties['id']);
		$notice->rawHtml($properties['content']);
		$notice->type($properties['noticeType']);
		$notice->addClass($properties['customCssClasses']);

		if ($properties['isDismissible']) {
			$notice->dismissible();
		}
		if ($properties['isPersistentlyDismissible']) {
			$notice->persistentlyDismissible();
		}

		return $notice;
	}

	/**
	 * Process an AJAX request to dismiss this notice.
	 * @internal
	 */
	public function ajaxDismiss() {
		check_ajax_referer($this->getDismissActionName());

		if (!is_user_logged_in()) {
			wp_die('Access denied. You need to be logged in to dismiss notices.');
			return;
		}

		$this->dismiss();
		exit('Notice dismissed');
	}

	public function dismiss() {
		if (!$this->isPersistentlyDismissible) {
			return;
		}
		update_option($this->getDismissOptionName(), true);
	}

	public function undismiss() {
		delete_option($this->getDismissOptionName());
	}

	/**
	 * Delete all "dismissed" flags that have the specified prefix.
	 *
	 * @param string $prefix
	 */
	public static function cleanUpDatabase($prefix) {
		global $wpdb; /** @var \wpdb $wpdb */

		$wpdb->query(sprintf(
			'DELETE FROM %s WHERE option_name LIKE "%s"',
			$wpdb->options,
			esc_sql($wpdb->esc_like(static::DISMISSED_OPTION_PREFIX . $prefix) . '%')
		));
	}

	public function isDismissed() {
		if (!$this->isPersistentlyDismissible) {
			return false;
		}
		return get_option($this->getDismissOptionName(), false);
	}

	protected function getDismissActionName() {
		return 'ye_dismiss-' . $this->id;
	}

	protected function getDismissOptionName() {
		return static::DISMISSED_OPTION_PREFIX . $this->id;
	}
}

add_action('admin_notices', array(__NAMESPACE__ . '\AdminNotice', '_showDelayedNotices'));

/**
 * Create an admin notice.
 *
 * @param string $id
 * @return AdminNotice
 */
function easyAdminNotice($id) {
	return new AdminNotice($id);
}